/*
** FCD treating routines
**
** Writen by Sakae Tatibana <tatibana@extra.hu>
**
** 1999, 8/26 coding started for compress FCD treating
** 2000, 1/17 modified for multi data track FCD
**       3/16 modified for track edit & create FCD
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "binary.h"     /* basic binary treating interfaces        */
#include "fcd_head.h"   /* FCD Header structures and interfaces    */
#include "fcd_decode.h" /* FCD compress decodeing routine          */ 
#include "filename.h"   /* read_filename()                         */
#include "mp3_info.h"   /* MP3 Info structures and interfaces      */
#include "riff_wav.h"   /* RIFF/WAV Info structures and interfaces */
#include "iso_xa.h"     /* ISO/XA Info structures and interfaces   */
#include "edc_ecc.h"

#define FCD_C
#include "fcd.h"

int open_fcd_file(char *in, FCD *out);
int read_fcd_info(FILE *in, FCD *out);
int close_fcd_file(FCD *fcd);
int read_fcd_first_blocks(FCD *in, cd_image_buffer *out);
int read_fcd_data(FCD *in, cd_image_buffer *out);
int go_next_track(FCD *in);
int write_fcd_info(FCD *in, FILE *out);
int write_fcd_data(cd_image_buffer *in, FCD *out);
int init_fcd(FCD *fcd);
int add_track(FCD *fcd, char *filename);
int del_track(FCD *fcd, int track);
int move_track(FCD *fcd, int track, int direction);
int sort_track_by_name(FCD *fcd);
int read_offset_table(FCD *in, OFFSET_TABLE *out);
static void add_data_track_list(FCD *fcd, int track, int compress, size_t offset);
static basic_fcd_header *FCD_to_BFH(FCD *in);
static multi_track_extantion_header *FCD_to_MTEH(FCD *in);
static int add_track_da_wav(FCD *fcd, char *filename);
static int add_track_da_mp3(FCD *fcd, char *filename);
static int add_track_iso(FCD *fcd, char *filename);
static void swap_ti(cd_infomation *cdi, int index);
static void swap_dt(data_track_list *dtl, int index);
static void swap_at(audio_track_list *atl, int index);
static void set_start_and_gap(track_infomation *ti);
static int add_track_unknown(FCD *fcd, char *filename);
static data_track *next_blank_data_track(FCD *fcd);
static audio_track *next_blank_audio_track(FCD *fcd);
static track_infomation *next_blank_track_infomation(FCD *fcd);

static const unsigned char sync[SYNC_SIZE] = {
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0x00,
};

static const LBN post_gap[5][5] = {
	{  0,    0,    0,    0,    0,},
	{  0,  150,  150,    0,  150,},
	{  0,  150,  150,    0,  150,},
	{  0,    0,    0,    0,    0,},
	{  0,    1,    1,    0,    1,},
};

/********************************************************************
    Open existing FCD file.
********************************************************************/
int open_fcd_file(char *in, FCD *out)
{
	out->path = in;
	out->stream = fopen(in, "rb");

	if(out->stream == NULL){
		return 0;
	}

	if(read_fcd_info(out->stream, out)){
		return 1;
	}else{
		fclose(out->stream);
		return 0;
	}
}

/********************************************************************
    Close FCD File
********************************************************************/
int close_fcd_file(FCD *fcd)
{
	fclose(fcd->stream);
	return 1;
}

/********************************************************************
   Read stream and set fcd infomation.

   This function has a side efect. The stream is set position
   to head of first track.
********************************************************************/
int read_fcd_info(FILE *in, FCD *out)
{
	int i;

	basic_fcd_header *bfh;
	singl_track_extantion_header *steh;
	multi_track_extantion_header *mteh;
	recordable_extantion_header *reh;

	int is_recordable = 0;

	bfh = get_basic_header(in);
	if(bfh == NULL){
		return 0;
	}
	out->pos = FBH_SIZE;

	out->stream = in;
    strcpy(out->cdi.desc, bfh->desc);

	out->dtl.num = 0;
	out->dtl.current = out->dtl.dt;

	out->atl.num = 0;
	out->atl.current = out->atl.at;

	if(bfh->fcd_type == 5){
        mteh = get_mte_header(in);
        if(mteh == NULL){
            free(bfh);
            return 0;
        }

		out->pos += MTEH_SIZE;

		out->cdi.num_of_track = mteh->track[1];
		out->cdi.total = mteh->total;
		out->type = FCD_TYPE_UNKNOWN;

		if(mteh->tt[0].type & 0x40){
			/****************************************************************
			                 Fucking FCD Spec Infomation

			   Only in the first track, data track LBN & MSF is added 16.

               Remove it in here.
			 ****************************************************************/

			mteh->tt[0].start.f = 0;
            mteh->ti[0].start = 0;
		}

		/****************************************************************
		                  Fucking FCD Spec Infomation

		   mteh->track[] & mteh->total value trustworthy is NOTING.

           they lose reason. their behavior can't be predicted.

		   Check all track for multi session FCD file.
		 ****************************************************************/
		for(i=0;i<TRACK_MAX && mteh->tt[i].type;i++){

            out->cdi.ti[i].num_of_block = mteh->ti[i].end - mteh->ti[i].start;
            out->cdi.ti[i].index = i;
            out->cdi.ti[i].start = mteh->ti[i].start;

            out->type |= FCD_TYPE_ISO;
            out->cdi.ti[i].type = FCD_TYPE_ISO;
            if(i<TRACK_MAX -1 && mteh->tt[i+1].type){
                out->cdi.ti[i].post_gap = mteh->ti[i+1].start - mteh->ti[i].end;
            }else{
                out->cdi.ti[i].post_gap = post_gap[FCD_TYPE_ISO][FCD_TYPE_UNKNOWN];
            }
            out->cdi.ti[i].block_size = 2048;
            strcpy(out->cdi.ti[i].filename, "");

			switch(mteh->ti[i].type){
			case 0:
                add_data_track_list(out, i, FCD_COMPRESS_NO, mteh->ti[i].data_offset);
				break;
			case 1:
                add_data_track_list(out, i, FCD_COMPRESS_STD, mteh->ti[i].data_offset);
				break;
			case 2:
				out->type |= FCD_TYPE_DA;
				out->cdi.ti[i].type = FCD_TYPE_DA;
                if(i<TRACK_MAX -1 && mteh->tt[i+1].type){
                    out->cdi.ti[i].post_gap = mteh->ti[i+1].start - mteh->ti[i].end;
                }else{
					out->cdi.ti[i].post_gap = post_gap[FCD_TYPE_DA][FCD_TYPE_UNKNOWN];
                }
                out->cdi.ti[i].block_size = 0;

                strcpy(out->cdi.ti[i].filename, out->path);
                cut_filename(out->cdi.ti[i].filename);
                strcat(out->cdi.ti[i].filename, mteh->ti[i].filename);

                out->atl.current = out->atl.at + out->atl.num;
                out->atl.current->index = out->atl.num;

                out->cdi.ti[i].p = (void *) out->atl.current;

                out->atl.current->ti = out->cdi.ti + i;
                out->atl.current->byte = 0;

                out->atl.num += 1;

				break;
			case 3:
				out->type |= FCD_TYPE_RAW;
				out->cdi.ti[i].type = FCD_TYPE_RAW;
                out->cdi.ti[i].block_size = 2352;

                add_data_track_list(out, i, FCD_COMPRESS_NO, mteh->ti[i].data_offset);
				break;
			case 5:
                add_data_track_list(out, i, FCD_COMPRESS_HI, mteh->ti[i].data_offset);
				break;
			default:
				out->cdi.ti[i].type = FCD_TYPE_UNKNOWN;
                if(i<TRACK_MAX-1 && mteh->tt[i+1].type){
                    out->cdi.ti[i].post_gap = mteh->ti[i+1].start - mteh->ti[i].end;
                }else{
                    out->cdi.ti[i].post_gap = post_gap[FCD_TYPE_UNKNOWN][FCD_TYPE_UNKNOWN];
                }
                out->cdi.ti[i].block_size = 0;
            }
        }

		/****************************************************************
		                  Fucking FCD Spec Infomation

		   mteh->track[] & mteh->total value trustworthy is NOTING.

		   Update these data in here.
		 ****************************************************************/
        out->cdi.num_of_track = i;
        out->cdi.total = LBN_to_MSF(out->cdi.ti[i-1].start + out->cdi.ti[i-1].num_of_block + out->cdi.ti[i-1].post_gap);

		if(out->dtl.num){
			file_skip(in, (out->dtl.dt[0].offset - out->pos));
            out->pos = out->dtl.dt[0].offset;
        }
		free(mteh);

    }else{
        steh = get_ste_header(in);
		if(steh == NULL){
			free(bfh);
			return 0;
		}
		out->pos += STEH_SIZE;

        if(!(steh->first)){
            steh->total = bfh->fcd_size / 2048;
        }else{

            /******************************************************************
                                 Fucking FCD Spec Infomation

             Single track (uncompressed) FCD contains total MSF in LBN format.
             (include 2sec gap)

			 And teamQQQ's ISO2FCD sets correct LBN (exclude 2sec gap) same
			 point.
		 
            ******************************************************************/

			if(steh->total != (bfh->fcd_size / 2048)){
				steh->total -= 150;
			}
        }

        out->cdi.num_of_track = 1;
        out->cdi.total = LBN_to_MSF(steh->total);
        out->cdi.ti[0].index = 0;
        out->cdi.ti[0].start = 0;
        out->cdi.ti[0].num_of_block = steh->total;
        out->cdi.ti[0].post_gap = post_gap[FCD_TYPE_ISO][FCD_TYPE_UNKNOWN];
        out->cdi.ti[0].block_size = 2048;
        strcpy(out->cdi.ti[0].filename, "");

        out->type = FCD_TYPE_ISO;
        out->cdi.ti[0].type = FCD_TYPE_ISO;

        switch(bfh->fcd_type) {
        case 0:
            add_data_track_list(out, 0, FCD_COMPRESS_NO, STEH_OFSET + STEH_SIZE);
			break;
		case 1:
			add_data_track_list(out, 0, FCD_COMPRESS_STD, STEH_OFSET + STEH_SIZE);
			break;
		case 2:
			add_data_track_list(out, 0, FCD_COMPRESS_NO, REH_OFSET + REH_SIZE);
			is_recordable = 1;
			break;
		case 3:
			add_data_track_list(out, 0, FCD_COMPRESS_STD, REH_OFSET + REH_SIZE);
			is_recordable = 1;
			break;
		case 6:
			add_data_track_list(out, 0, FCD_COMPRESS_HI, STEH_OFSET + STEH_SIZE);
			break;
		case 8:
			add_data_track_list(out, 0, FCD_COMPRESS_HI, REH_OFSET + REH_SIZE);
			is_recordable = 1;
			break;
		default:
			out->type = FCD_TYPE_UNKNOWN;
			out->cdi.ti[0].type = FCD_TYPE_UNKNOWN;
		}

		free(steh);

		if(is_recordable){
            reh = get_re_header(in);
			if(reh == NULL){
                free(bfh);
				return 0;
			}
			free(reh);
			out->pos += REH_SIZE;
		}
	}

    free(bfh);

    out->dtl.current = out->dtl.dt;
    out->atl.current = out->atl.at;

	if(out->dtl.num && out->dtl.dt[0].compress > 0){
		if(is_recordable){
            read_le_int32(in, &i);
			out->pos += 4;
		}

		if(!read_le_int32(in, &i)){
			return 0;
        }
		out->pos += 4;

		file_skip(in, i - out->pos);
		out->pos = i;
	}else if(is_recordable){
		file_skip(in, 32768);
		out->pos += 32768;
	}

	return 1;
}

/********************************************************************
   Read FCD data (include first 16 blocks) to cd_image_buffer

   return value is same as out->num_of_block
********************************************************************/
int read_fcd_first_blocks(FCD *in, cd_image_buffer *out)
{
	int i;
	size_t n;

	if(in->dtl.current->ti->index){ /* without first track */
		out->mode = 0; /* unspecified. probably 2 */
		return read_fcd_data(in, out);
	}

	if(out->block_size == 2048){
		out->data = (char *)calloc(2048 * 16, 1);
		out->num_of_block = 16;
		out->mode = 1;
        in->dtl.current->position += 16;
		return 16;
	}

	out->data = (char *)calloc(2352 * 32, 1);

	n = fread(out->data + (2352 * 16), 1, (2352 * 16), in->stream);
	if(n == 0){
		return 0;
	}

	out->num_of_block = 16 + (n / 2352);
    in->dtl.current->position += out->num_of_block;
    in->pos += n;

	out->mode = out->data[2352 * 16 + SYNC_SIZE + 3]; /* mm: 1, ss: 1, ff: 1 */

	if(out->mode != 1 && out->mode != 2){
		if(memcmp(out->data + (2352 * 16 + SYNC_SIZE + 4 + 1), "CD001", 5)){
			out->mode = 2;
		}else{
			out->mode = 1;
		}
	}

	for(i=0;i<16;i++){
		memcpy(out->data + (2352 * i), sync, SYNC_SIZE);

		out->data[2352 * i + SYNC_SIZE + 0] = 0;
		out->data[2352 * i + SYNC_SIZE + 1] = 2;
		out->data[2352 * i + SYNC_SIZE + 2] = binary_to_4bit_decimal(i) & 0xFF;
		out->data[2352 * i + SYNC_SIZE + 3] = out->mode;

		if(out->mode == 1){
			/* It is a lazy work, sure.
			   but I don't know how to calc EDC & ECC */
			memcpy(out->data + (2352 * i + 12 + 4 + 2048), edc_ecc[i], 288);
		}
	}

	return out->num_of_block;
}

/********************************************************************
   Read FCD data to cd_image_buffer

   return value is same as out->num_of_block
********************************************************************/
int read_fcd_data(FCD *in, cd_image_buffer *out)
{
	size_t n;
	LBN block;

	block = in->dtl.current->ti->num_of_block - in->dtl.current->position;
	if(block > 16){
		block = 16;
	}

	switch(in->dtl.current->compress){
	case FCD_COMPRESS_NO:
		out->data = (char *)calloc(out->block_size * block, 1);
		n = fread(out->data, 1, out->block_size * block, in->stream);
		in->pos += n;
		out->num_of_block = n / out->block_size;
        in->dtl.current->position += out->num_of_block;
        break;
	case FCD_COMPRESS_STD:
		in->dtl.current->position += fcd_std_unit_decode(in, out, NULL);
        break;
    case FCD_COMPRESS_HI:
        in->dtl.current->position += fcd_hi_unit_decode(in, out, NULL);
        break;
	default:
		return 0;
	}

    if(in->dtl.current->ti->num_of_block < in->dtl.current->position){
        out->num_of_block -= (in->dtl.current->position - in->dtl.current->ti->num_of_block);
    }

    return out->num_of_block;
}

/********************************************************************
   Set FCD stream position to head of next track
********************************************************************/
int go_next_track(FCD *in)
{
	int pos;

	in->dtl.current += 1;

	pos = ftell(in->stream);
	if(pos < 0){ /* stream is pipe? */
		pos = in->pos;
	}else if(pos != in->pos){
		/* the code has bug(s)!! check! check! check! */
		in->pos = pos;
	}
	file_skip(in->stream, in->dtl.current->offset - pos);
	in->pos = in->dtl.current->offset;

	if(in->dtl.current->compress > 0){
		if(!read_le_int32(in->stream, &pos)){
			return 0;
		}
		in->pos += 4;

		file_skip(in->stream, pos - in->pos);
		in->pos = pos;
	}

	return 1;
}

/********************************************************************
   Write FCD Infomation to stream

   This function has a side efect. The stream is set position
   to end of the FCD header.
********************************************************************/
int write_fcd_info(FCD *in, FILE *out)
{
	basic_fcd_header *bfh;
	multi_track_extantion_header *mteh;

	bfh = FCD_to_BFH(in);
	if(bfh == NULL){
		return 0;
	}
	if(!write_basic_header(bfh, out)){
		free(bfh);
		return 0;
	}
	free(bfh);
	in->pos = FBH_SIZE;

	mteh = FCD_to_MTEH(in);
	if(mteh == NULL){
		return 0;
	}
	if(!write_mte_header(mteh, out)){
		free(mteh);
		return 0;
	}
	free(mteh);
	in->pos += MTEH_SIZE;

	return 1;
}

/********************************************************************
   Write DATA from cd_image_buffer

   return value is same as in->num_of_block
********************************************************************/
int write_fcd_data(cd_image_buffer *in, FCD *out)
{
	int i;
	size_t n;
	LBN block;

    MSF msf;

	block = in->num_of_block;
	if(block > 16){
		block = 16;
	}

	if(out->dtl.current->compress != FCD_COMPRESS_NO){
		return 0;
    }

    for(i=0;i<block;i++){
        if(in->block_size == 2336){
            out->pos += fwrite(sync, 1, SYNC_SIZE, out->stream);

            msf = LBN_to_MSF(out->dtl.current->ti->start + out->dtl.current->position);

            fputc((binary_to_4bit_decimal(msf.m) & 0xFF), out->stream);
            fputc((binary_to_4bit_decimal(msf.s) & 0xFF), out->stream);
            fputc((binary_to_4bit_decimal(msf.f) & 0xFF), out->stream);
            fputc(in->mode, out->stream);

            out->pos += 4;
        }

		n = fwrite(in->data+(i*in->block_size), 1, in->block_size, out->stream);
		out->pos += n;
        if(n != in->block_size){
            in->num_of_block = i;
            return i;
        }
        out->dtl.current->position += 1;
	}

    in->num_of_block = i;
    return in->num_of_block;
}

/********************************************************************
   Initialize FCD structure

   return value is 1 (success) / 0 (fail)
********************************************************************/
int init_fcd(FCD *fcd)
{
	int i;

    if(fcd == NULL){
        return 0;
    }

	fcd->path = NULL;
	fcd->stream = NULL;
	fcd->pos = 0;
	fcd->type = FCD_TYPE_UNKNOWN;

    for(i=0;i<DESC_MAX;i++){
        fcd->cdi.desc[i] = '\0';
    }
    fcd->cdi.num_of_track = 0;
    fcd->cdi.total.padding = 0;
    fcd->cdi.total.m = 0;
    fcd->cdi.total.s = 0;
    fcd->cdi.total.f = 0;

    for(i=0;i<TRACK_MAX;i++){
        fcd->cdi.ti[i].index = i;
        fcd->cdi.ti[i].type = FCD_TYPE_UNKNOWN;
        fcd->cdi.ti[i].start = 0;
        fcd->cdi.ti[i].num_of_block = 0;
        fcd->cdi.ti[i].post_gap = post_gap[FCD_TYPE_UNKNOWN][FCD_TYPE_UNKNOWN];
        fcd->cdi.ti[i].block_size = 0;
        strcpy(fcd->cdi.ti[i].filename, "");
        fcd->cdi.ti[i].p = NULL;
    }

    fcd->dtl.num = 0;
    fcd->dtl.current = fcd->dtl.dt;

    for(i=0;i<TRACK_MAX;i++){
        fcd->dtl.dt[i].index = i;
        fcd->dtl.dt[i].ti = NULL;
        fcd->dtl.dt[i].compress = FCD_COMP_UNKNOWN;
        fcd->dtl.dt[i].offset = 0x2776;
		fcd->dtl.dt[i].mode = 0;
		fcd->dtl.dt[i].iso_xa = 0;
        fcd->dtl.dt[i].position = 0;
    }

    fcd->atl.num = 0;
    fcd->atl.current = fcd->atl.at;

    for(i=0;i<TRACK_MAX;i++){
        fcd->atl.at[i].index = i;
        fcd->atl.at[i].ti = NULL;
        fcd->atl.at[i].byte = 0;
    }

    return 1;
}

/********************************************************************
   Add track at FCD

   return value is 1 (success) / 0 (fail)
********************************************************************/
int add_track(FCD *fcd, char *filename)
{
    if(fcd->cdi.num_of_track >= TRACK_MAX){
        return 0;
    }

	if(check_sfx(filename, ".wav")){
		return add_track_da_wav(fcd, filename);
	}else if(check_sfx(filename, ".mp3")){
		return add_track_da_mp3(fcd, filename);
	}else if(add_track_iso(fcd, filename)){
		return 1;	
	}else{
		return add_track_unknown(fcd, filename);
	}
}

/********************************************************************
   Delete track

   return value is 1 (success) / 0 (fail)
********************************************************************/
int del_track(FCD *fcd, int track)
{
	int i;

	while(move_track(fcd, track, 1)){
		track += 1;
	}

	fcd->cdi.num_of_track -= 1;

	fcd->cdi.total = LBN_to_MSF(fcd->cdi.ti[track].start - fcd->cdi.ti[track-1].post_gap);

    if(fcd->cdi.ti[track].type == FCD_TYPE_DA){
        fcd->atl.num -= 1;
        fcd->atl.at[fcd->atl.num].ti = NULL;
        fcd->atl.at[fcd->atl.num].byte = 0;
    }else{
        fcd->dtl.num -= 1;
        fcd->dtl.dt[fcd->dtl.num].ti = NULL;
        fcd->dtl.dt[fcd->dtl.num].compress = FCD_COMP_UNKNOWN;
        fcd->dtl.dt[fcd->dtl.num].offset = 0x2776;
        fcd->dtl.dt[fcd->dtl.num].position = 0;
    }

	fcd->cdi.ti[track].type = FCD_TYPE_UNKNOWN;

	fcd->cdi.ti[track].start = 0;
	fcd->cdi.ti[track].num_of_block = 0;
	fcd->cdi.ti[track].post_gap = post_gap[FCD_TYPE_UNKNOWN][FCD_TYPE_UNKNOWN];
	fcd->cdi.ti[track].block_size = 0;

	for(i=0;i<FILENAME_MAX;i++){
		fcd->cdi.ti[track].filename[i] = '\0';
	}

    fcd->cdi.ti[track].p = NULL;

	return 1;
}

/********************************************************************
   Move track UP/DOWN

   return value is 1 (success) / 0 (fail)
********************************************************************/
int move_track(FCD *fcd, int track, int down)
{
	int i;
	int flag;
    audio_track *at;
    data_track *dt;

    if(down){
        return move_track(fcd, track+1, 0);
    }

    if(track < 1 || track >= fcd->cdi.num_of_track){
        return 0;
    }

    flag = 0;

    if(fcd->cdi.ti[track].type == FCD_TYPE_DA){
        at = (audio_track *)(fcd->cdi.ti[track].p);
        at->ti -= 1;
        flag += 1;
    }else{
        dt = (data_track *)(fcd->cdi.ti[track].p);
        dt->ti -= 1;
        flag -= 1;
    }

    if(fcd->cdi.ti[track-1].type == FCD_TYPE_DA){
        at = (audio_track *)(fcd->cdi.ti[track-1].p);
        at->ti += 1;
        flag += 1;
    }else{
        dt = (data_track *)(fcd->cdi.ti[track-1].p);
        dt->ti += 1;
        flag -= 1;
    }

    swap_ti(&(fcd->cdi), track);

    if(flag){
        if(fcd->cdi.ti[track].type == FCD_TYPE_DA){
            at = (audio_track *)(fcd->cdi.ti[track-1].p);
            swap_at(&(fcd->atl), at->index);
            fcd->cdi.ti[track-1].p = fcd->cdi.ti[track].p;
            fcd->cdi.ti[track].p = (void *)at;
        }else{
            dt = (data_track *)(fcd->cdi.ti[track-1].p);
            swap_dt(&(fcd->dtl), dt->index);
            fcd->cdi.ti[track-1].p = fcd->cdi.ti[track].p;
            fcd->cdi.ti[track].p = (void *)dt;
        }
	}

	for(i=track-1;i<fcd->cdi.num_of_track;i++){
		set_start_and_gap(fcd->cdi.ti+i);
	}
	
	return 1;
}

/********************************************************************
   sort track by fcd->cdi.filename[] (ignore suffix)

   return value is move count
********************************************************************/
int sort_track_by_name(FCD *fcd)
{
    int i,d;
    int r;
    char *b1, *b2;

    r = 0;

    if(fcd == NULL){
        return 0;
    }

    if(fcd->cdi.num_of_track < 2){
        return 0;
    }

    for(i=0;i<fcd->cdi.num_of_track-1;i++){
        b1 = get_basename(fcd->cdi.ti[i].filename);
        b2 = get_basename(fcd->cdi.ti[i+1].filename);
        d = strcmp(b1,b2);

        if(d>0){
            move_track(fcd, i, 1);
            r += 1;
            i -= 2;
        }

        free(b1);
        free(b2);

        if(i<-1){
            i = 0;
        }
    }

    return r;
}

/********************************************************************
   read compressed image offset table (1st data track only)

   return value is same as out->num_of_units;
********************************************************************/
int get_offset_table(FCD *in, OFFSET_TABLE *out)
{
	int i;
	
	if(in->dtl.dt[0].compress == FCD_COMPRESS_NO){
		out->num_of_unit = 0;
		return 0;
	}

	fseek(in->stream, in->dtl.dt[0].offset, SEEK_SET);

	out->num_of_unit = (in->dtl.dt[0].ti->num_of_block + 15) / 16;
	out->offsets = (size_t *)malloc(out->num_of_unit*sizeof(size_t));

	if(in->dtl.dt[0].offset == REH_OFSET + REH_SIZE){
		read_le_int32(in->stream, (int *)(out->offsets));
	}

	for(i=0;i<out->num_of_unit;i++){
		read_le_int32(in->stream, (int *)(out->offsets+i));
	}

	fseek(in->stream, out->offsets[0], SEEK_SET);
	
	return out->num_of_unit;
}

static void add_data_track_list(FCD *fcd, int track, int compress, size_t offset)
{
    fcd->dtl.current = fcd->dtl.dt + fcd->dtl.num;
    fcd->dtl.current->index = fcd->dtl.num;

    fcd->cdi.ti[track].p = (void *)(fcd->dtl.current);

    fcd->dtl.current->ti = fcd->cdi.ti + track;
    fcd->dtl.current->compress = compress;
    fcd->dtl.current->offset = offset;
    fcd->dtl.current->position = 0;

    fcd->dtl.num += 1;
}

static basic_fcd_header *FCD_to_BFH(FCD *in)
{
	int i;
	basic_fcd_header *bfh;

	bfh = (basic_fcd_header *)calloc(1,sizeof(basic_fcd_header));

	bfh->fcd_type = 5; /* Multi Track */

	switch(in->type){
	case FCD_TYPE_ISO:
		bfh->orig_type = 0;
		break;
	case FCD_TYPE_RAW:
		bfh->orig_type = 5;
		break;
	case (FCD_TYPE_RAW + FCD_TYPE_ISO):
		bfh->orig_type = 6;
		break;
	case FCD_TYPE_DA:
		bfh->orig_type = 2;
		break;
	case (FCD_TYPE_DA + FCD_TYPE_ISO):
		bfh->orig_type = 1;
		break;
	case (FCD_TYPE_DA + FCD_TYPE_RAW):
	case (FCD_TYPE_DA + FCD_TYPE_RAW + FCD_TYPE_ISO):
		bfh->orig_type = 6;
		break;
	default:
		free(bfh);
		return NULL;
	}

	if(in->type & FCD_TYPE_RAW){
		bfh->trk_offset = 0x9300;
	}else{
		bfh->trk_offset = 0x8000;
	}

	bfh->fcd_size = 0;
	for(i=0;i<in->cdi.num_of_track;i++){
        switch(in->cdi.ti[i].type){
        case FCD_TYPE_ISO:
            bfh->fcd_size += (2048 * in->cdi.ti[i].num_of_block);
            break;
		case FCD_TYPE_RAW:
		case FCD_TYPE_DA:
			bfh->fcd_size += (2352 * in->cdi.ti[i].num_of_block);
            break;
        default:
            free(bfh);
            return NULL;
		}
	}

	strcpy(bfh->desc, in->cdi.desc);
	strcpy(bfh->sig, SIGNATURE);

    if(in->dtl.num){
        bfh->data_track_start = in->dtl.dt[0].ti->start;
    }else{
        bfh->data_track_start = 0;
    }

	return bfh;
}

static multi_track_extantion_header *FCD_to_MTEH(FCD *in)
{
	int i;
	multi_track_extantion_header *mteh;
	data_track *d;

	mteh = (multi_track_extantion_header *)calloc(1,sizeof(multi_track_extantion_header));

    if(in->dtl.num){
        mteh->track[0] = (in->dtl.dt[0].ti->index + 1) & 0xFF;
    }else{
        mteh->track[0] = 1;
    }
    if(mteh->track[0] > 1){
        mteh->track[1] = mteh->track[0] - 1;
    }else{
        mteh->track[1] = in->cdi.num_of_track & 0xFF;
    }
	mteh->total = in->cdi.total;

	for(i=0;i<in->cdi.num_of_track;i++){
		mteh->tt[i].start = LBN_to_MSF(in->cdi.ti[i].start);
		switch(in->cdi.ti[i].type){
		case FCD_TYPE_ISO:
		case FCD_TYPE_RAW:
			mteh->tt[i].type = 0x41;
			break;
		case FCD_TYPE_DA:
			mteh->tt[i].type = 0x01;
			break;
		default:
			free(mteh);
			return NULL;
		}
	}

	for(i=0;i<in->cdi.num_of_track;i++){
		mteh->ti[i].start = in->cdi.ti[i].start;
		mteh->ti[i].end = in->cdi.ti[i].start + in->cdi.ti[i].num_of_block;
		switch(in->cdi.ti[i].type){
		case FCD_TYPE_ISO:
			d = (data_track *)in->cdi.ti[i].p;
			switch(d->compress){
			case FCD_COMPRESS_NO:
				mteh->ti[i].type = 0;
				break;
			case FCD_COMPRESS_STD:
				mteh->ti[i].type = 1;
				break;
			case FCD_COMPRESS_HI:
				mteh->ti[i].type = 5;
				break;
			default:
				mteh->ti[i].type = 6;
			}
			break;
		case FCD_TYPE_RAW:
			mteh->ti[i].type = 3;
			break;
		case FCD_TYPE_DA:
			mteh->ti[i].type = 2;
    		strcpy(mteh->ti[i].filename, read_filename(in->cdi.ti[i].filename));
            cut_sfx(mteh->ti[i].filename);
            strcat(mteh->ti[i].filename, ".wav");
		    mteh->ti[i].data_offset = -1;
			break;
		default:
			free(mteh);
			return NULL;
		}
	}

	for(i=0;i<in->dtl.num;i++){
		mteh->ti[in->dtl.dt[i].ti->index].data_offset = in->dtl.dt[i].offset;
	}

	if(in->cdi.ti[0].type != FCD_TYPE_DA){

		/****************************************************************
			             Fucking FCD Spec Infomation

		   Only in the first track, data track LBN & MSF is added + 16

           Restore it in here.
		 ****************************************************************/

		mteh->tt[0].start.f += 16;
		mteh->ti[0].start += 16;
	}

	return mteh;
}

static int add_track_da_wav(FCD *fcd, char *filename)
{
	WAV_INFO wi;
    track_infomation *ti;
    audio_track *at;

	if(!open_wav_file(filename, &wi)){
		return 0;
	}

	ti = next_blank_track_infomation(fcd);

    fcd->type |= FCD_TYPE_DA;
    ti->type = FCD_TYPE_DA;

	set_start_and_gap(ti);
	
    ti->num_of_block = (wi.data_length / wi.byte_per_sec * 75) + ((wi.data_length % wi.byte_per_sec) * 75 / wi.byte_per_sec); /* avoid from overflow */
    ti->block_size = wi.byte_per_sec / 75;

	strcpy(ti->filename, filename);

    at = next_blank_audio_track(fcd);

    ti->p = (void *)at;

    at->ti = ti;
    at->byte = wi.file_length;

	close_wav_file(&wi);

    fcd->cdi.total = LBN_to_MSF(ti->start + ti->num_of_block + ti->post_gap);

	return 1;
}

static int add_track_da_mp3(FCD *fcd, char *filename)
{
	MP3_INFO mp3;
    track_infomation *ti;
    audio_track *at;

	if(!open_mp3_file(filename, &mp3)){
		return 0;
	}

	ti = next_blank_track_infomation(fcd);

    fcd->type |= FCD_TYPE_DA;
    ti->type = FCD_TYPE_DA;

	set_start_and_gap(ti);
	
    ti->num_of_block = (mp3.length * 3) / (mp3.bitrate * 5);
    if((mp3.length * 3) % (mp3.bitrate * 5)){
        ti->num_of_block += 1;
    }
    ti->block_size = mp3.bitrate * 5 / 3;
    strcpy(ti->filename, filename);

    at = next_blank_audio_track(fcd);

    ti->p = (void *)at;

    at->ti = ti;
    at->byte = mp3.length;

	close_mp3_file(&mp3);

    fcd->cdi.total = LBN_to_MSF(ti->start + ti->num_of_block + ti->post_gap);

	return 1;
}

static int add_track_iso(FCD *fcd, char *filename)
{
	ISO_INFO iso;
    track_infomation *ti;
    data_track *dt;

	if(!open_iso_file(filename, &iso)){
		return 0;
	}

	ti = next_blank_track_infomation(fcd);

 	switch(iso.block_size){
	case 2048:
        fcd->type |= FCD_TYPE_ISO;
		ti->type = FCD_TYPE_ISO;
		break;
	case 2336:
	case 2352:
        fcd->type |= FCD_TYPE_RAW;
        ti->type = FCD_TYPE_RAW;
		break;
	default:
		close_iso_file(&iso);
		return 0;
	}
	
	ti->start = iso.start;
	set_start_and_gap(ti);
	
    ti->num_of_block = iso.num_of_block;
    ti->block_size = iso.block_size;
    strcpy(ti->filename, filename);

    dt = next_blank_data_track(fcd);

    ti->p = (void *)dt;

    dt->ti = ti;
    dt->compress = FCD_COMPRESS_NO;
    dt->offset = 0x2776;
	dt->mode = iso.mode;
	dt->iso_xa = 1;
    dt->position = 0;

	strcpy(fcd->cdi.desc, iso.volume_identifier);

    close_iso_file(&iso);

	return 1;
}

static void swap_ti(cd_infomation *cdi, int index)
{
    track_infomation swap;

    swap = cdi->ti[index];
    cdi->ti[index] = cdi->ti[index-1];
    cdi->ti[index-1] = swap;

    cdi->ti[index].index = index;
    cdi->ti[index-1].index = index-1;
    
}

static void swap_dt(data_track_list *dtl, int index)
{
    data_track swap;

    swap = dtl->dt[index];
    dtl->dt[index] = dtl->dt[index-1];
    dtl->dt[index-1] = swap;

    dtl->dt[index].index = index;
    dtl->dt[index-1].index = index-1;
}

static void swap_at(audio_track_list *atl, int index)
{
    audio_track swap;

    swap = atl->at[index];
    atl->at[index] = atl->at[index-1];
    atl->at[index-1] = swap;

    atl->at[index].index = index;
    atl->at[index-1].index = index-1;
}

static void set_start_and_gap(track_infomation *ti)
{
	if(ti->index){
		if( (ti->type != FCD_TYPE_DA)  &&  ( ((data_track *)(ti->p))->iso_xa ) ){
			(ti-1)->post_gap = ti->start - (ti-1)->start - (ti-1)->num_of_block;
		}else{
			(ti-1)->post_gap = post_gap[(ti-1)->type][ti->type];
			ti->start = (ti-1)->start + (ti-1)->num_of_block + (ti-1)->post_gap;
		}
	}else if(ti->type == FCD_TYPE_DA){
        ti->start = 0;
    }
}

static int add_track_unknown(FCD *fcd, char *filename)
{
	FILE *in;
	
	int n;
	size_t length;

	size_t block_size;
	int mode;

	track_infomation *ti;
	data_track *dt;

	in = fopen(filename, "rb");
	if(in == NULL){
		return 0;
	}

	fseek(in, 0, SEEK_SET);
	n = ftell(in);
	fseek(in, 0, SEEK_END);
	length = ftell(in) - n;

	fclose(in);

	if((length % 2048) == 0){
		block_size = 2048;
		mode = 1;
	}else if(length % 2336){
		block_size = 2336;
		mode = 2;
	}else if(length % 2352){
		block_size = 2352;
		mode = 2;
	}else{
		return 0;
	}

	ti = next_blank_track_infomation(fcd);

 	switch(block_size){
	case 2048:
        fcd->type |= FCD_TYPE_ISO;
		ti->type = FCD_TYPE_ISO;
		break;
	case 2336:
	case 2352:
        fcd->type |= FCD_TYPE_RAW;
        ti->type = FCD_TYPE_RAW;
		break;
	default:
		return 0;
	}

	strcpy(ti->filename, filename);

    ti->num_of_block = length / block_size;
    ti->block_size = block_size;

    dt = next_blank_data_track(fcd);

    ti->p = (void *)dt;

    dt->ti = ti;
    dt->compress = FCD_COMPRESS_NO;
    dt->offset = 0x2776;
	dt->mode = mode;
	dt->iso_xa = 0;
    dt->position = 0;
	
	set_start_and_gap(ti);

	return 1;
}

static data_track *next_blank_data_track(FCD *fcd)
{
	data_track *dt;
	
    dt = fcd->dtl.dt + fcd->dtl.num;
    dt->index = fcd->dtl.num;
    fcd->dtl.num += 1;

	return dt;
}

static audio_track *next_blank_audio_track(FCD *fcd)
{
	audio_track *at;
	
    at = fcd->atl.at + fcd->atl.num;
    at->index = fcd->atl.num;
    fcd->atl.num += 1;

	return at;
}

static track_infomation *next_blank_track_infomation(FCD *fcd)
{
	track_infomation *ti;
	
	ti = fcd->cdi.ti + fcd->cdi.num_of_track;
    ti->index = fcd->cdi.num_of_track;
	fcd->cdi.num_of_track += 1;

	return ti;
}